#include "common/samplers11.hlsl"
#include "common/states11.hlsl"
#include "common/context.hlsl"
#include "common/ambientCube.hlsl"
#include "common/softParticles.hlsl"
#include "common/random.hlsl"

#define CLOUDS_COLOR
#include "ParticleSystem2/common/psCommon.hlsl"
#include "ParticleSystem2/common/perlin.hlsl"
#include "ParticleSystem2/common/noiseSimplex.hlsl"

#ifdef MSAA
	Texture2DMS<float, MSAA> distanceMap;
#else
	Texture2D<float> distanceMap;
#endif

Texture2D	afterburnerTex;
Texture2D	noiseTex;
Texture2D	glowTex;

uint2		dims;
float		params2;
float4		params;
float4x4	World;

#define circleCount params.x
#define circleCountInv	params.y
#define time		params.z
#define power		params.w
#define emitterId	params2.x

static const float4	glowColor = float4(1, 0.7, 0.3, 1);

static const int	maxSegments = 26;
static const int	maxSegmentsHotAir = 8;

static const float	opacityFactor = 1.0f/maxSegments;
static const float	opacityFactorHotAir = 1.0f/maxSegmentsHotAir;

static const float	alphaOpacity = 0.4;

static const float	hotAirPower = 0.05;
static const float	hotAirDistOffset = 10.0;
static const float	hotAirDistMaxInv = 1.0 / 200.0;

static const float2 xLocal = {-0.13, 1.0};//      

SamplerState BlackBorderLinearSampler
{
	Filter        = MIN_MAG_MIP_LINEAR;
	AddressU      = BORDER;
	AddressV      = BORDER;
	AddressW      = BORDER;
	MaxAnisotropy = MAXANISOTROPY_DEFAULT;
	BorderColor   = float4(0, 0, 0, 0);
};


struct VS_OUTPUT {
	float4 pos		: POSITION0;
};

struct VS_CIRCLE_OUTPUT {
	uint vertId		: TEXCOORD2;
};

struct HS_CONST_OUTPUT {
	float edges[2]	: SV_TessFactor;
};

struct HS_OUTPUT{
	float4 pos		: POSITION0;
};

struct DS_OUTPUT{
	float4 pos		: POSITION0;
};

struct GS_OUTPUT{
	float4 pos		: SV_POSITION0;
	float4 projPos	: TEXCOORD0;
	float4 UV		: TEXCOORD1;
	float  cldAlpha	: TEXCOORD2;
	float  dist		: TEXCOORD3;
};


VS_OUTPUT vs(uint vertId: SV_VertexId)
{
	VS_OUTPUT o;
	o.pos = 0;
	return o;
}
VS_CIRCLE_OUTPUT vsCircle(uint vertId: SV_VertexId)
{
	VS_CIRCLE_OUTPUT o;
	o.vertId = vertId;
	return o;
}

float GetAfterburnerAttenuation(float attPower = 1.0)
{
	return 1 - attPower * max(0,gSunDir.y);
}

// HULL SHADER ---------------------------------------------------------------------
HS_CONST_OUTPUT hsConstant( InputPatch<VS_OUTPUT, 1> ip, uint pid : SV_PrimitiveID )
{
	HS_CONST_OUTPUT o;
	o.edges[1] = maxSegments;
	o.edges[0] = 1; 
	return o;
}
HS_CONST_OUTPUT hsConstantHotAir( InputPatch<VS_OUTPUT, 1> ip, uint pid : SV_PrimitiveID )
{
	HS_CONST_OUTPUT o;
	o.edges[1] = maxSegmentsHotAir;
	o.edges[0] = 1; 
	return o;
}
[domain("isoline")]
[partitioning("integer")]
[outputtopology("point")]
[outputcontrolpoints(1)]
[patchconstantfunc("hsConstant")]
HS_OUTPUT hs(InputPatch<VS_OUTPUT, 1> ip, uint cpid : SV_OutputControlPointID)
{
	HS_OUTPUT o;
	o.pos = ip[0].pos;
	return o;
}

[domain("isoline")]
[partitioning("integer")]
[outputtopology("point")]
[outputcontrolpoints(1)]
[patchconstantfunc("hsConstantHotAir")]
HS_OUTPUT hsHotAir(InputPatch<VS_OUTPUT, 1> ip, uint cpid : SV_OutputControlPointID)
{
	HS_OUTPUT o;
	o.pos = ip[0].pos;
	return o;
}

// DOMAIN SHADER ---------------------------------------------------------------------
[domain("isoline")]
DS_OUTPUT ds( HS_CONST_OUTPUT input, float2 UV : SV_DomainLocation, const OutputPatch<HS_OUTPUT, 1> patch )
{
	DS_OUTPUT o;
	float2 sc;
	sincos(UV.x*PI, sc.x, sc.y);
	o.pos = float4(sc*0.55, opacityFactor, 0);
	return o; 
}

// CIRCLE ---------------------------------
float getCircleParamStutter(float param)
{
	return  0.1 + 0.5*param + noise2D(float2(time*1.52, param*5.123+emitterId*0.3639))*0.007*(param+0.4);
}
float2 getCircleUV(in float2 UV, float t)
{
	UV.x *= 20;//
	UV.x -= 20*t;	
	// t *= t;	
	// UV.y *= 1 + t*1.6;//
	// UV.y -= t*0.8;	
	
	// UV.y *= 0.85 + t*2;//
	UV.y *= 0.85 + t*2;//
	UV.y -= t - 0.075;
	
	return UV;
}

void generateCircle(inout TriangleStream<GS_OUTPUT> outputStream, inout GS_OUTPUT o, in float xPos, in float param, uniform bool bHotAir)
{
	float4x4 WVP = mul(World, gViewProj);
	[unroll]
	for (int i = 0; i < 4; ++i)
	{
		float4 pos = {xPos, staticVertexData[i].xy, 1};
		if(bHotAir)
		{
			o.UV.xy = pos.yz * 2;
		}
		else
		{
			o.UV.xy = pos.yz + 0.5;
			o.UV.x = 0.875 + 0.125*o.UV.x;	// 64/512
		}
		pos.yz *= 1 - 0.5*param;//     
		
		o.pos = o.projPos = mul(pos, WVP);
		outputStream.Append(o);
	}
	outputStream.RestartStrip();
}
//-----------------------------------------

[maxvertexcount(4)]
void gs(point DS_OUTPUT input[1], inout TriangleStream<GS_OUTPUT> outputStream, uniform bool bHotAir, uniform bool bClouds)
{
	#define sinCos input[0].pos.xy
	#define gsOpacity input[0].pos.z

	GS_OUTPUT o;
	
	float3 pos = {-xLocal.x, sinCos.x, sinCos.y};
	float3 pos2 = {-xLocal.x, -sinCos.x, -sinCos.y};
	
	float3 normal = cross(normalize(pos), float3(1,0,0));
	normal = mul(float4(normal,0), World).xyz;
	
	// o.UV.z = gsOpacity * pow(abs(dot(normal, gView._13_23_33)), 1);
	o.UV.z = pow(abs(dot(normal, gView._13_23_33)), 0.3);
	o.UV.z *= 0.4 +0.6*gsOpacity;
	o.UV.w = sinCos.x;
	
	float4 vPos = mul(mul(float4(0,0,0, 1), World), gView); vPos /= vPos.w;
	o.dist = max(0, vPos.z) * 1.73 / gProj._11;
	
	if(bClouds)
		o.cldAlpha = 1-getCloudsColor(0).a;
	else
		o.cldAlpha = 1;
	
	float4x4 WVP = mul(World, gViewProj);

	// 1
	o.pos = o.projPos = mul(float4(pos,1), WVP);
	o.UV.xy = float2(0,0);
	outputStream.Append(o);
	o.pos = o.projPos = mul(float4(pos2,1), WVP);
	o.UV.xy = float2(0,1);
	outputStream.Append(o);
	
	// 2
	pos.x = pos2.x = -xLocal.y*0.9;//0.9       X
	o.pos = o.projPos = mul(float4(pos,1), WVP);
	o.UV.xy = float2(0.9,0);
	outputStream.Append(o);
	o.pos = o.projPos = mul(float4(pos2,1), WVP);
	o.UV.xy = float2(0.9,1);
	outputStream.Append(o);

	outputStream.RestartStrip();
}

// static const uint slices = 1;

[maxvertexcount(4)]
void gsCircle(point VS_CIRCLE_OUTPUT input[1], inout TriangleStream<GS_OUTPUT> outputStream, uniform bool bHotAir, uniform bool bClouds)
{
	#define gsVertId input[0].vertId.x
	GS_OUTPUT o;
	
	float t = getCircleParamStutter(uint((float)gsVertId+0.1f)*circleCountInv);

	float att = GetAfterburnerAttenuation(0.4);

	float4 vPos = mul(mul(float4(0,0,0, 1), World), gView); vPos /= vPos.w;
	o.dist = max(0, vPos.z) * 1.73 / gProj._11;
	
	if(bClouds)
		o.UV.z = (1-t) * att * (1-getCloudsColor(0).a);
	else
		o.UV.z = (1-t) * att;
	
	o.UV.w = 0;
	o.cldAlpha = 0;
	
	float param = (xLocal.x+0.012) + (xLocal.y-xLocal.x)*t;	
	generateCircle(outputStream, o, -param, t, bHotAir);
}

void depthTest(float2 projPos, float depthRef)
{
	float2 uv = projPos.xy*0.5 + 0.5;
	uv.y = 1-uv.y;
#ifdef MSAA
	float depth = distanceMap.Load(uint2(uv*dims), 0).r;
#else
	float depth = distanceMap.Load(int3(uv*dims, 0)).r;
#endif
	clip(depthRef-depth);
}

float4 ps(in GS_OUTPUT i, uniform bool bHotAir): SV_TARGET0
{	
	float psOpacity = i.UV.z;
	
	float2 UVbase = i.UV.xy;
	i.UV.x *= 0.875*1.111;//0.875 is the ratio of afterburner flame length / whole texture length, b/c we have the circle on the right of the AB texture (using 64x64px)   (512-64)/512
	i.UV.x *= 0.72 + 0.2*saturate(noise1D(time+emitterId*1.3264 +i.UV.w*6.1232));
	
	float4 baseColor = afterburnerTex.Sample(WrapLinearSampler, i.UV.xy);
	

	// MOD COMMENT - lerp(valueA, valueB, vector) : linear interpolation between valueA and valueB. Vector should be in range [0;1], a vector value of 0 will set valueA, and a vector value of 1 will set valueB
	// MOD COMMENT - saturate(x)	: return 0 if x<0, return 1 if x>1, return x if x=[0;1]. We'll use this function to make sure a value will remain in range [0;1]
	// MOD COMMENT - pow(x,y) 		: return x to the power of y. This function is known to have issues with null values, so if you want to use integer powers, you'd better do (eg. for x^5:) x*x*x*x*x...
	
	// MOD COMMENT - basecolor.rgb :
	// MOD COMMENT - This will change afterburner texture color depending on sun attenuation. A brighter sun will make a more yellow/orange AB. It will gradually fade to purple with sun going down.
	// MOD COMMENT - RGB color values are in range [0.00;1.00]. Multiply this value by 255 to convert it to common RGB color range.
	// MOD COMMENT - gSunAttenuation seems to be in range [0;1], starts to sweep from 0 to 1 at early sundown, so we'll use power to "retard" the effect. Can't use pow() as a null or negative value will result in errors.
	// MOD COMMENT - gSunIntensity function is very "steep". Value will remain high till teh exact moment of sundown and goes to 0 in a few minutes. Range unknown, seems to reach values [0;10+], so we have to use saturate() to avoid negative opacity values.
	
	// MOD COMMENT - IF YOUR WANT TO CHANGE THE DAY COLOR : you have to change values in the first float3(R,G,B). Make sure all values are in range [0.0;1.0]
	// MOD COMMENT - IF YOUR WANT TO CHANGE THE NIGHT COLOR : you have to change values in the second float3(R,G,B). Make sure all values are in range [0.0;1.0]
	baseColor.rgb += lerp(float3(0,0,0), float3(0.7, 0.6, 1.0), (gSunAttenuation * gSunAttenuation * gSunAttenuation * gSunAttenuation * gSunAttenuation))*0.7;
	// MOD COMMENT - Opacity will depend of gSunAttenuation and gSunIntensity. 
	baseColor.a *= (0.7 + (0.3 * gSunAttenuation * gSunAttenuation * gSunAttenuation * gSunAttenuation * (1.0 - saturate(gSunIntensity * 0.1))));
	// MOD COMMENT - Sorry, I insist. Do the maths, and make sure you WILL NEVER GET NEGATIVE VALUES for baseColor.a. THAT'S UGLY. 
	// MOD COMMENT - at 1200 we'll have 	(0.2+0.8*0) * (1-0.4*saturate(10*0.1)) = 0.2*0.6 = 0.12... Supposed gSunIntensity will be =10 at max intensity
	// MOD COMMENT - at 0000 we'll have		(0.2+0.8*1) * (1-0.4*saturate( 0*0.1)) = 1.0*1.0 = 1.0 ... If you have an opacity value > 1 you may end up with a "hard texture clip" in the end of the AB trail.
	
	
	// MOD COMMENT - Default values : 
//	baseColor.rgb += lerp(pow(float3(0.89,0.54,0.47)*1.2,2), float3(0.39, 0.32, 0.85), i.UV.x)*0.6; // MOD COMMENT - This line was commented in vanilla DCS file
//	baseColor.a *= alphaOpacity ;	


	if(bHotAir)
	{
		depthTest(i.projPos.xy/i.projPos.w, i.projPos.z/i.projPos.w);
		return float4(1,1,1, min(1, baseColor.a * baseColor.a * i.cldAlpha * hotAirPower * psOpacity * 2.5 * saturate(1 - (i.dist-hotAirDistOffset) * hotAirDistMaxInv)));
	}
	else
	{
		float noiseMask = baseColor.a;
		
		baseColor.a *= psOpacity * 0.45 ;
		baseColor.a *= GetAfterburnerAttenuation(0.5);
		
		// (circles)
		const uint nCircles = (uint)circleCount;
		for(uint j=0; j<nCircles; ++j)
		{
			float t = getCircleParamStutter(j*circleCountInv);
			float4 circle = tex.Sample(BlackBorderLinearSampler, getCircleUV(UVbase, t));
			baseColor += circle*0.03;
		}
		
		// simplex noise
		// float2 noiseUV = UVbase*5; 
		// noiseUV.x *= 1.5;
		// noiseUV.y += i.UV.w*0.3523;
		// noiseUV += i.UV.w*1.521-time*30;
		// float noise = snoise(noiseUV);

		// (pearls ?)
		float2 noiseUV = UVbase*0.4*0.3; 
		noiseUV.x *= 1.5;
		noiseUV.y += i.UV.w*0.1523;
		noiseUV.xy += i.UV.w*0.5128 + emitterId*4.1264 - time*2;
		float noise = noiseTex.Sample(WrapLinearSampler, noiseUV*2).r;

		noise = lerp(noise, 0.00, saturate( (1-i.UV.x) + pow(sin(i.UV.y*PI), 5) ) );
		// return noise;
		
		// MOD COMMENT - Noise added to the AB opacity. First line was active, and ADD noise to the AB opacity layer. Second line was inactive (commented) and MULTIPLY noise by AB opacity layer.
		//baseColor.a += noise*0.5 * pow(noiseMask,1.5);
		// baseColor.a *= (1 + noise*1 * pow(noiseMask,1.5));
		
		// MOD COMMENT - So... Adding noise isn't a good idea, as we'll make opaque some parts we want to be fully transparent. 
		// MOD COMMENT - We'll use the noiseMask (our AB opacity layer) as a basis, then we'll add some noise on that, and multiply this noise by our noisemask. That way, we're sure the fully transparent parts in the texture WILL ALWAYS REMAIN fully transparent.
		baseColor.a *= (2 * noiseMask * noiseMask + 0.5 * noise * saturate(4 * noiseMask -1));
		


		return baseColor * baseColor * i.cldAlpha * 2;
	}
}

float4 psCircle(in GS_OUTPUT i, uniform bool bHotAir): SV_TARGET0
{
	if(bHotAir)
	{
		depthTest(i.projPos.xy/i.projPos.w, i.projPos.z/i.projPos.w);
		float alpha = max(0, 1 - dot(i.UV.xy, i.UV.xy));
		return float4(1, 1, 1, min(alpha*2, 1) * hotAirPower * 0.8 * saturate(1 - (i.dist-hotAirDistOffset) * hotAirDistMaxInv));
	}
	else
	{
		float cParam  = i.UV.w * 0.2;
		float4 clr = afterburnerTex.Sample(WrapLinearSampler, i.UV.xy);
		clr.a *= alphaOpacity;
		clr.a = max(0,clr.a-cParam) / (1-cParam);

		return clr * i.UV.z * 0.72;
	}
}

struct PS_INPUT
{
	float4 pos: SV_POSITION0;
	float3 uv : TEXCOORD0;
};

static const float4 quad2[4] = {
	float4( -0.5, -0.5, 0, 0),
	float4( -0.5,  0.5, 0, 0),
	float4(  0.5, -0.5, 0, 0),
	float4(  0.5,  0.5, 0, 0)
};

PS_INPUT vsGlow(uint vertId:  SV_VertexID)
{
	PS_INPUT o;
	float rnd = frac(sin(gModelTime*321513.5123));
	float4 vPos = quad2[vertId];
	
	const float scale = 1.1;
	const float opacityMax = 0.7;
	
	o.pos = mul(mul(float4(-0.15,0,0,1), World), gView);
	float dist = max(0, o.pos.z)*1.73/gProj._11;
	float scaleFactor = scale.x * (1 + 0.0005 * dist);
	o.pos += vPos * scaleFactor * (5 + rnd);
	o.pos = mul(o.pos, gProj);
	
	float visibiltyFactor = saturate(3*(dot(normalize(World._11_12_13), gView._31_32_33)));
	visibiltyFactor = sqrt(visibiltyFactor) * min(1, dist/600);
	
	//stuttering
	float2 sc;
	sincos( smoothNoise1(gModelTime*10+rnd*0.1)*6.2832, sc.x, sc.y );
	o.uv.xy = mul(vPos.xy, float2x2(sc.y, sc.x, -sc.x, sc.y)) + 0.5;
	o.uv.z = visibiltyFactor * opacityMax * (0.9 + 0.1*rnd);
	
	return o;
}

float4 psGlow(PS_INPUT i, uniform bool bClouds): SV_TARGET0
{
	float4 color = lerp(float4(1, 0.7, 0.3, 1), float4(0.4, 0.3, 1, 1), gSunAttenuation * gSunAttenuation) * glowTex.Sample(ClampLinearSampler, i.uv.xy).r * i.uv.z;
	// float4 color = glowColor * glowColor * glowTex.Sample(ClampLinearSampler, i.uv.xy).r * i.uv.z;
	
	if(bClouds)
		color *= 1-sbCloudsColor[cloudColorId.x].a;
	
	return color;
}


technique10 Textured
{
	pass afterburner
	{
		SetVertexShader(CompileShader(vs_4_0, vs()));
		SetHullShader(CompileShader(hs_5_0, hs()));
		SetDomainShader(CompileShader(ds_5_0, ds()));
		SetGeometryShader(CompileShader(gs_4_0, gs(false, false)));
		SetPixelShader(CompileShader(ps_4_0, ps(false)));
		
		SetBlendState(additiveAlphaBlend, float4(0.0f, 0.0f, 0.0f, 0.0f), 0xFFFFFFFF);
		SetDepthStencilState(enableDepthBufferNoWrite, 0);
		SetRasterizerState(cullNone);
	}
	pass circles
	{
		SetVertexShader(CompileShader(vs_4_0, vsCircle()));	
		SetHullShader(NULL);
		SetDomainShader(NULL);
		SetGeometryShader(CompileShader(gs_4_0, gsCircle(false, false)));
		SetPixelShader(CompileShader(ps_4_0, psCircle(false)));
		
		SetBlendState(additiveAlphaBlend, float4(0.0f, 0.0f, 0.0f, 0.0f), 0xFFFFFFFF);
		SetDepthStencilState(enableDepthBufferNoWrite, 0);
		SetRasterizerState(cullNone);
	}
	
	pass glow
	{
		SetVertexShader(CompileShader(vs_4_0, vsGlow()));
		SetHullShader(NULL);
		SetDomainShader(NULL);
		SetGeometryShader(NULL);
		SetPixelShader(CompileShader(ps_4_0, psGlow(false)));
		
		SetBlendState(additiveAlphaBlend, float4(0.0f, 0.0f, 0.0f, 0.0f), 0xFFFFFFFF);
		SetDepthStencilState(enableDepthBufferNoWrite, 0);
		SetRasterizerState(cullNone);
	}	
	
	pass afterburnerClouds
	{
		SetVertexShader(CompileShader(vs_4_0, vs()));
		SetHullShader(CompileShader(hs_5_0, hs()));
		SetDomainShader(CompileShader(ds_5_0, ds()));
		SetGeometryShader(CompileShader(gs_4_0, gs(false, true)));
		SetPixelShader(CompileShader(ps_4_0, ps(false)));
		
		SetBlendState(additiveAlphaBlend, float4(0.0f, 0.0f, 0.0f, 0.0f), 0xFFFFFFFF);
		SetDepthStencilState(enableDepthBufferNoWrite, 0);
		SetRasterizerState(cullNone);
	}
	pass circlesClouds
	{
		SetVertexShader(CompileShader(vs_4_0, vsCircle()));	
		SetHullShader(NULL);
		SetDomainShader(NULL);
		SetGeometryShader(CompileShader(gs_4_0, gsCircle(false, true)));
		SetPixelShader(CompileShader(ps_4_0, psCircle(false)));
		
		SetBlendState(additiveAlphaBlend, float4(0.0f, 0.0f, 0.0f, 0.0f), 0xFFFFFFFF);
		SetDepthStencilState(enableDepthBufferNoWrite, 0);
		SetRasterizerState(cullNone);
	}
	
	pass glowClouds
	{
		SetVertexShader(CompileShader(vs_4_0, vsGlow()));
		SetHullShader(NULL);
		SetDomainShader(NULL);
		SetGeometryShader(NULL);
		SetPixelShader(CompileShader(ps_4_0, psGlow(true)));
		
		SetBlendState(additiveAlphaBlend, float4(0.0f, 0.0f, 0.0f, 0.0f), 0xFFFFFFFF);
		SetDepthStencilState(enableDepthBufferNoWrite, 0);
		SetRasterizerState(cullNone);
	}
	
	pass afterburnerHotAir
	{
		SetVertexShader(CompileShader(vs_4_0, vs()));
		SetHullShader(CompileShader(hs_5_0, hsHotAir()));
		SetDomainShader(CompileShader(ds_5_0, ds()));
		SetGeometryShader(CompileShader(gs_4_0, gs(true, false)));
		SetPixelShader(CompileShader(ps_4_0, ps(true)));
		
		SetBlendState(additiveAlphaBlend, float4(0.0f, 0.0f, 0.0f, 0.0f), 0xFFFFFFFF);
		SetDepthStencilState(enableDepthBufferNoWrite, 0);
		SetRasterizerState(cullNone);
	}
	pass circlesHotAir
	{
		SetVertexShader(CompileShader(vs_4_0, vsCircle()));	
		SetHullShader(NULL);
		SetDomainShader(NULL);
		SetGeometryShader(CompileShader(gs_4_0, gsCircle(true, false)));
		SetPixelShader(CompileShader(ps_4_0, psCircle(true)));
		
		SetBlendState(additiveAlphaBlend, float4(0.0f, 0.0f, 0.0f, 0.0f), 0xFFFFFFFF);
		SetDepthStencilState(enableDepthBufferNoWrite, 0);
		SetRasterizerState(cullNone);
	}
}
